use crate::{
    config::Config,
    enhance::ScriptType,
    utils::{dirs, help},
};
use anyhow::{bail, Context, Result};
use reqwest::StatusCode;
use serde::{Deserialize, Serialize};
use serde_yaml::Mapping;
use std::fs;
use sysproxy::Sysproxy;
use tracing_attributes::instrument;

use super::item_type::{ProfileItemType, ProfileUid};

#[derive(Debug, Clone, Deserialize, Serialize)]
pub struct ProfileItem {
    pub uid: Option<String>,

    /// profile item type
    /// enum value: remote | local | script | merge
    #[serde(rename = "type")]
    pub r#type: Option<ProfileItemType>,

    /// profile name
    pub name: Option<String>,

    /// profile file
    pub file: Option<String>,

    /// profile description
    #[serde(skip_serializing_if = "Option::is_none")]
    pub desc: Option<String>,

    /// source url
    #[serde(skip_serializing_if = "Option::is_none")]
    pub url: Option<String>,

    /// selected information
    #[serde(skip_serializing_if = "Option::is_none")]
    pub selected: Option<Vec<PrfSelected>>,

    /// subscription user info
    #[serde(skip_serializing_if = "Option::is_none")]
    pub extra: Option<PrfExtra>,

    /// updated time
    pub updated: Option<usize>,

    /// some options of the item
    #[serde(skip_serializing_if = "Option::is_none")]
    pub option: Option<PrfOption>,

    /// the file data
    #[serde(skip)]
    pub file_data: Option<String>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub chains: Option<Vec<ProfileUid>>, // Save the profile relates profile chains. The String should be the uid of the profile.
}

impl Default for ProfileItem {
    fn default() -> Self {
        ProfileItem {
            uid: None,
            r#type: Some(ProfileItemType::Local),
            name: None,
            file: None,
            desc: None,
            url: None,
            selected: None,
            extra: None,
            updated: None,
            option: None,
            file_data: None,
            chains: None,
        }
    }
}

#[derive(Default, Debug, Clone, Deserialize, Serialize)]
pub struct PrfSelected {
    pub name: Option<String>,
    pub now: Option<String>,
}

#[derive(Default, Debug, Clone, Copy, Deserialize, Serialize)]
pub struct PrfExtra {
    pub upload: usize,
    pub download: usize,
    pub total: usize,
    pub expire: usize,
}

#[derive(Default, Debug, Clone, Deserialize, Serialize, PartialEq, Eq)]
pub struct PrfOption {
    /// for `remote` profile's http request
    /// see issue #13
    #[serde(skip_serializing_if = "Option::is_none")]
    pub user_agent: Option<String>,

    /// for `remote` profile
    /// use system proxy
    #[serde(skip_serializing_if = "Option::is_none")]
    pub with_proxy: Option<bool>,

    /// for `remote` profile
    /// use self proxy
    #[serde(skip_serializing_if = "Option::is_none")]
    pub self_proxy: Option<bool>,

    #[serde(skip_serializing_if = "Option::is_none")]
    pub update_interval: Option<u64>,
}

impl PrfOption {
    pub fn merge(one: Option<Self>, other: Option<Self>) -> Option<Self> {
        match (one, other) {
            (Some(mut a), Some(b)) => {
                a.user_agent = b.user_agent.or(a.user_agent);
                a.with_proxy = b.with_proxy.or(a.with_proxy);
                a.self_proxy = b.self_proxy.or(a.self_proxy);
                a.update_interval = b.update_interval.or(a.update_interval);
                Some(a)
            }
            t => t.0.or(t.1),
        }
    }
}

impl ProfileItem {
    /// From partial item
    /// must contain `itype`
    pub async fn duplicate(item: ProfileItem, file_data: Option<String>) -> Result<ProfileItem> {
        match item.r#type {
            Some(ProfileItemType::Remote) => {
                if item.url.is_none() {
                    bail!("url should not be null");
                }
                let url = item.url.as_ref().unwrap().as_str();
                let name = item.name.unwrap_or("Remote File".into());
                let desc = item.desc.unwrap_or("".into());
                ProfileItem::from_url(url, Some(name), Some(desc), item.option).await
            }
            Some(ProfileItemType::Local) => {
                let name = item.name.unwrap_or("Local File".into());
                let desc = item.desc.unwrap_or("".into());
                ProfileItem::from_local(name, desc, file_data)
            }
            Some(ProfileItemType::Merge) => {
                let name = item.name.unwrap_or("Merge".into());
                let desc = item.desc.unwrap_or("".into());
                ProfileItem::from_merge(name, desc, file_data)
            }
            Some(ProfileItemType::Script(script_type)) => {
                let name = item.name.unwrap_or("Script".into());
                let desc = item.desc.unwrap_or("".into());
                ProfileItem::from_script(name, desc, script_type, file_data)
            }
            None => bail!("could not find the item type"),
        }
    }

    /// ## Local type
    /// create a new item from name/desc
    pub fn from_local(
        name: String,
        desc: String,
        file_data: Option<String>,
    ) -> Result<ProfileItem> {
        let uid = help::get_uid("l");
        let file = format!("{uid}.yaml");

        Ok(ProfileItem {
            uid: Some(uid),
            r#type: Some(ProfileItemType::Local),
            name: Some(name),
            desc: Some(desc),
            file: Some(file),
            updated: Some(chrono::Local::now().timestamp() as usize),
            file_data,
            ..Default::default()
        })
    }

    /// ## Remote type
    /// create a new item from url
    #[instrument]
    pub async fn from_url(
        url: &str,
        name: Option<String>,
        desc: Option<String>,
        option: Option<PrfOption>,
    ) -> Result<ProfileItem> {
        let opt_ref = option.as_ref();
        let with_proxy = opt_ref.map_or(false, |o| o.with_proxy.unwrap_or(false));
        let self_proxy = opt_ref.map_or(false, |o| o.self_proxy.unwrap_or(false));
        let user_agent = opt_ref.and_then(|o| o.user_agent.clone());

        let mut builder = reqwest::ClientBuilder::new().use_rustls_tls().no_proxy();

        // 使用软件自己的代理
        if self_proxy {
            let port = Config::verge()
                .latest()
                .verge_mixed_port
                .unwrap_or(Config::clash().data().get_mixed_port());

            let proxy_scheme = format!("http://127.0.0.1:{port}");

            if let Ok(proxy) = reqwest::Proxy::http(&proxy_scheme) {
                builder = builder.proxy(proxy);
            }
            if let Ok(proxy) = reqwest::Proxy::https(&proxy_scheme) {
                builder = builder.proxy(proxy);
            }
            if let Ok(proxy) = reqwest::Proxy::all(&proxy_scheme) {
                builder = builder.proxy(proxy);
            }
        }
        // 使用系统代理
        else if with_proxy {
            if let Ok(p @ Sysproxy { enable: true, .. }) = Sysproxy::get_system_proxy() {
                let proxy_scheme = format!("http://{}:{}", p.host, p.port);

                if let Ok(proxy) = reqwest::Proxy::http(&proxy_scheme) {
                    builder = builder.proxy(proxy);
                }
                if let Ok(proxy) = reqwest::Proxy::https(&proxy_scheme) {
                    builder = builder.proxy(proxy);
                }
                if let Ok(proxy) = reqwest::Proxy::all(&proxy_scheme) {
                    builder = builder.proxy(proxy);
                }
            };
        }

        let version = dirs::get_app_version();
        let version = format!("clash-nyanpasu/v{version}");
        builder = builder.user_agent(user_agent.unwrap_or(version));

        let resp = builder.build()?.get(url).send().await?;

        let status_code = resp.status();
        if !StatusCode::is_success(&status_code) {
            bail!("failed to fetch remote profile with status {status_code}")
        }

        let header = resp.headers();
        tracing::debug!("headers: {:#?}", header);

        // parse the Subscription UserInfo
        let extra = match header
            .get("subscription-userinfo")
            .or(header.get("Subscription-Userinfo"))
        {
            Some(value) => {
                tracing::debug!("Subscription-Userinfo: {:?}", value);
                let sub_info = value.to_str().unwrap_or("");

                Some(PrfExtra {
                    upload: help::parse_str(sub_info, "upload").unwrap_or(0),
                    download: help::parse_str(sub_info, "download").unwrap_or(0),
                    total: help::parse_str(sub_info, "total").unwrap_or(0),
                    expire: help::parse_str(sub_info, "expire").unwrap_or(0),
                })
            }
            None => None,
        };

        // parse the Content-Disposition
        let filename = match header
            .get("content-disposition")
            .or(header.get("Content-Disposition"))
        {
            Some(value) => {
                tracing::debug!("Content-Disposition: {:?}", value);

                let filename = format!("{value:?}");
                let filename = filename.trim_matches('"');
                match help::parse_str::<String>(filename, "filename*") {
                    Some(filename) => {
                        let iter = percent_encoding::percent_decode(filename.as_bytes());
                        let filename = iter.decode_utf8().unwrap_or_default();
                        filename.split("''").last().map(|s| s.to_string())
                    }
                    None => match help::parse_str::<String>(filename, "filename") {
                        Some(filename) => {
                            let filename = filename.trim_matches('"');
                            Some(filename.to_string())
                        }
                        None => None,
                    },
                }
            }
            None => None,
        };

        // parse the profile-update-interval
        let option = match header
            .get("profile-update-interval")
            .or(header.get("Profile-Update-Interval"))
        {
            Some(value) => {
                tracing::debug!("profile-update-interval: {:?}", value);
                match value.to_str().unwrap_or("").parse::<u64>() {
                    Ok(val) => Some(PrfOption {
                        update_interval: Some(val * 60), // hour -> min
                        ..PrfOption::default()
                    }),
                    Err(_) => None,
                }
            }
            None => None,
        };

        let uid = help::get_uid("r");
        let file = format!("{uid}.yaml");
        let name = name.unwrap_or(filename.unwrap_or("Remote File".into()));
        let data = resp.text_with_charset("utf-8").await?;

        // process the charset "UTF-8 with BOM"
        let data = data.trim_start_matches('\u{feff}');

        // check the data whether the valid yaml format
        let yaml = serde_yaml::from_str::<Mapping>(data)
            .context("the remote profile data is invalid yaml")?;

        if !yaml.contains_key("proxies") && !yaml.contains_key("proxy-providers") {
            bail!("profile does not contain `proxies` or `proxy-providers`");
        }

        Ok(ProfileItem {
            uid: Some(uid),
            r#type: Some(ProfileItemType::Remote),
            name: Some(name),
            desc,
            file: Some(file),
            url: Some(url.into()),
            extra,
            option,
            updated: Some(chrono::Local::now().timestamp() as usize),
            file_data: Some(data.into()),
            ..Default::default()
        })
    }

    /// ## Merge type (enhance)
    /// create the enhanced item by using `merge` rule
    pub fn from_merge(
        name: String,
        desc: String,
        file_data: Option<String>,
    ) -> Result<ProfileItem> {
        let uid = help::get_uid("m");
        let file = format!("{uid}.yaml");

        Ok(ProfileItem {
            uid: Some(uid),
            r#type: Some(ProfileItemType::Merge),
            name: Some(name),
            desc: Some(desc),
            file: Some(file),
            updated: Some(chrono::Local::now().timestamp() as usize),
            file_data,
            ..Default::default()
        })
    }

    /// ## Script type (enhance)
    /// create the enhanced item by using javascript quick.js
    pub fn from_script(
        name: String,
        desc: String,
        script_type: ScriptType,
        file_data: Option<String>,
    ) -> Result<ProfileItem> {
        let uid = help::get_uid("s");
        let file = match script_type {
            ScriptType::JavaScript => format!("{uid}.js"), // js ext
            ScriptType::Lua => format!("{uid}.lua"),       // lua ext
        }; // js ext

        Ok(ProfileItem {
            uid: Some(uid),
            r#type: Some(ProfileItemType::Script(script_type)),
            name: Some(name),
            desc: Some(desc),
            file: Some(file),
            updated: Some(chrono::Local::now().timestamp() as usize),
            file_data,
            ..Default::default()
        })
    }

    /// get the file data
    pub fn read_file(&self) -> Result<String> {
        if self.file.is_none() {
            bail!("could not find the file");
        }

        let file = self.file.clone().unwrap();
        let path = dirs::app_profiles_dir()?.join(file);
        fs::read_to_string(path).context("failed to read the file")
    }

    /// save the file data
    pub fn save_file(&self, data: String) -> Result<()> {
        if self.file.is_none() {
            bail!("could not find the file");
        }

        let file = self.file.clone().unwrap();
        let path = dirs::app_profiles_dir()?.join(file);
        fs::write(path, data.as_bytes()).context("failed to save the file")
    }
}
